"use client"

import { DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { TransformedAgencies } from "@/types/transformers";
import { ComponentWithClassName } from "@/types/utils";
import { zodResolver } from "@hookform/resolvers/zod";
import { useState } from "react";
import { useForm } from "react-hook-form";
import { z } from "zod";
import { updateStatewideService } from "../lib/actions";
import { TransformedStatewideService } from "../type/transformers";
import { statewideServiceFormDataSchema } from "../zod/schema.form";
import { ErrorNoAgenciesContent } from "./error.no-agencies-content";
import { WorkflowFormStatewideServices } from "./workflow.form-statewide-services";
import { AppFormErrors } from "../type/errors";
import { revalidateGetStatewideServices } from "@/hooks/use-get-statewide-services";

export function WorkflowDialogUpdate({ setClose, data, agencies, className }: ComponentWithClassName<{ setClose: () => void, data: NonNullable<TransformedStatewideService>, agencies: TransformedAgencies }>) {
  const [errors, setErrors] = useState<AppFormErrors>(undefined)

  const form = useForm<z.infer<typeof statewideServiceFormDataSchema>>({
    resolver: zodResolver(statewideServiceFormDataSchema),
    defaultValues: {
      service_phone: data.phone ?? undefined,
      service_url: data.url ?? undefined,
      service_message: data.message,
      service_agency: data.agencyId,
      service_alias: data.alias,
    }
  })

  if (!agencies || agencies.length === 0) {
    return (
      <ErrorNoAgenciesContent />
    )
  }

  return (
    <DialogContent className={className}>
      <DialogHeader>
        <DialogTitle>Edit service details</DialogTitle>
        <DialogDescription>Currently updating details for &apos;<span className="font-mono">{data.alias}</span>&apos;.</DialogDescription>
      </DialogHeader>

      <WorkflowFormStatewideServices
        form={form}
        agencies={agencies}
        errors={errors}
        overrides={{
          cancelText: "Discard changes",
          submitText: "Edit service",
        }}
        handleSubmit={async (values) => {
          const res = await updateStatewideService({
            id: data.id,
            url: values.service_url ?? null,
            phone: values.service_phone ?? null,
            alias: values.service_alias,
            message: values.service_message,
            agencyid: values.service_agency,
            status: "PUBLISHED",
          })

          await revalidateGetStatewideServices()

          if (!!res.errors && res.errors.length > 0) {
            setErrors(res.errors.map((e) => e.message))
            return;
          }

          setClose()
        }}
      />
    </DialogContent >
  )
}

